<?php

namespace DMS\Includes\Integrations\Divi;

use DMS\Includes\Frontend\Frontend;
use DMS\Includes\Frontend\Mapping_Objects\Divi_Shop_Mapper;
use DMS\Includes\Frontend\Mapping_Objects\Shop_Mapper;
use DMS\Includes\Utils\Helper;

class Divi {

	/**
	 * Instance of the class
	 *
	 * @var Divi
	 */
	private static Divi $instance;

	/**
	 * Run the integration
	 *
	 * @return void
	 */
	public static function run(): void {
		add_action( 'dms_rewrite_uris', array( self::get_instance(), 'rewrite_uris' ) );
		add_action( 'dms_object_mapper', array( self::get_instance(), 'add_divi_shop_object_mapper' ), 10, 3 );
	}

	/**
	 * Get the singleton instance
	 *
	 * @return Divi
	 */
	public static function get_instance(): Divi {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Return divi shop page mapper if it runs shop page
	 *
	 * @param $class
	 * @param $mapping_value
	 * @param $query
	 *
	 * @return Divi_Shop_Mapper|Shop_Mapper|mixed
	 */
	public function add_divi_shop_object_mapper( $class, $mapping_value, $query ) {
		if ( $class instanceof Shop_Mapper && $this->divi_runs_shop_page() ) {
			return new Divi_Shop_Mapper( $mapping_value, $query );
		}

		return $class;
	}

	/**
	 * Check whether Divi supports the shop page
	 *
	 * @return bool
	 */
	public function divi_runs_shop_page(): bool {
		$shop_page_id = wc_get_page_id( 'shop' );
		// Check if Divi theme or parent theme is active
		return Helper::active_theme_is_divi() && function_exists( 'et_pb_is_pagebuilder_used' ) && et_pb_is_pagebuilder_used( $shop_page_id );
	}

	/**
	 * Rewrite uris
	 *
	 * @return void
	 */
	public function rewrite_uris(): void {
		$divi = self::get_instance();
		add_filter( 'et_builder_custom_fonts', array( $divi, 'rewrite_et_builder_custom_fonts' ) );
		add_filter( 'et_core_page_resource_tag', array( $divi, 'rewrite_et_core_page_resource_tag' ), 999, 5 );
		add_filter( 'et_pb_module_shortcode_attributes', array( $divi, 'rewrite_et_lottie_urls' ), 999, 5 );
	}

	/**
	 * Rewrite divi custom fonts
	 *
	 * @param $all_custom_fonts
	 *
	 * @return mixed
	 */
	public function rewrite_et_builder_custom_fonts( $all_custom_fonts ) {
		if ( is_array( $all_custom_fonts ) ) {
			foreach ( $all_custom_fonts as &$font ) {
				if ( ! empty( $font['font_url'] ) && is_array( $font['font_url'] ) ) {
					foreach ( $font['font_url'] as $sub_key => $font_url ) {
						$font['font_url'][ $sub_key ] = Frontend::get_instance()->uri_handler->replace_host_occurrence( $font_url );
					}
				}
			}
		}

		return $all_custom_fonts;
	}

	/**
	 * Rewrite link, style tags uris generated by Divi
	 *
	 * @param $tag
	 * @param $slug
	 * @param $scheme
	 * @param $onload
	 *
	 * @return string
	 */
	public function rewrite_et_core_page_resource_tag( $tag, $slug, $scheme, $onload ) {
		return Frontend::get_instance()->uri_handler->replace_host_occurrence( $tag );
	}

	/**
	 * Rewrite lottie urls (json)
	 *
	 * @param $props
	 * @param $attrs
	 * @param $render_slug
	 * @param $_address
	 * @param $content
	 *
	 * @return mixed
	 */
	public function rewrite_et_lottie_urls( $props, $attrs, $render_slug, $_address, $content ) {
		if ( ! empty( $props['lottie_url'] ) ) {
			$props['lottie_url'] = Frontend::get_instance()->uri_handler->replace_host_occurrence( $props['lottie_url'] );
		}

		return $props;
	}
}
